<?php
/**
 * TemplateConfigurationController.inc.php 2020-3-30
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

use Gambio\Core\Configuration\ConfigurationService;
use Gambio\Core\Configuration\Models\Write\Configuration;
use Gambio\StyleEdit\DependencyInjector;

MainFactory::load_class('HttpViewController');

/**
 * Class TemplateConfigurationController
 * @extends    HttpViewController
 * @category   System
 * @package    AdminHttpViewControllers
 */
class TemplateConfigurationController extends AdminHttpViewController
{
    /**
     * @var CI_DB_query_builder $db
     */
    protected $db;
    
    /**
     * @var LanguageTextManager $languageTextManager
     */
    protected $languageTextManager;
    
    /**
     * @var string
     */
    protected $shopEnvironment;
    
    /**
     * @var string
     */
    protected $styleEditLink;
    /**
     * @var string
     */
    protected $styleEdit3Link;
    /**
     * @var string
     */
    protected $styleEdit4Link;
    
    /**
     * @var ConfigurationService
     */
    private $configurationService;
    
    
    /**
     * @param HttpContextReaderInterface     $httpContextReader
     * @param HttpResponseProcessorInterface $httpResponseProcessor
     * @param ContentViewInterface           $contentView
     */
    public function __construct(
        HttpContextReaderInterface $httpContextReader,
        HttpResponseProcessorInterface $httpResponseProcessor,
        ContentViewInterface $contentView
    ) {
        parent::__construct($httpContextReader, $httpResponseProcessor, $contentView);
        $gxCoreLoader = MainFactory::create('GXCoreLoader', MainFactory::create('GXCoreLoaderSettings'));
        $this->db     = $gxCoreLoader->getDatabaseQueryBuilder();
        
        $this->languageTextManager = MainFactory::create('LanguageTextManager', 'template_configuration');
        
        $this->configurationService = LegacyDependencyContainer::getInstance()
            ->get(ConfigurationService::class);
    }
    
    
    /**
     * Returns the Template Configuration Page
     *
     * @return HttpControllerResponse|RedirectHttpControllerResponse
     */
    public function actionDefault()
    {
        $this->contentView->set_template_dir(DIR_FS_ADMIN . 'html/content/');
        $this->_checkEnvironment();
        
        $html = $this->_render('template_configuration.html',
                               [
                                   'TEMPLATE_SELECTION'                                => $this->templateSelection(),
                                   'MAIN_SHOW_QTY_INFO'                                => gm_get_conf('MAIN_SHOW_QTY_INFO')
                                                                                          === 'true',
                                   'MAIN_SHOW_ATTRIBUTES'                              => gm_get_conf('MAIN_SHOW_ATTRIBUTES')
                                                                                          === 'true',
                                   'MAIN_SHOW_GRADUATED_PRICES'                        => gm_get_conf('MAIN_SHOW_GRADUATED_PRICES')
                                                                                          === 'true',
                                   'SHOW_MANUFACTURER_IMAGE_PRODUCT_DETAILS'           => gm_get_conf('SHOW_MANUFACTURER_IMAGE_PRODUCT_DETAILS')
                                                                                          === 'true',
                                   'SHOW_ADDITIONAL_FIELDS_PRODUCT_DETAILS'            => gm_get_conf('SHOW_ADDITIONAL_FIELDS_PRODUCT_DETAILS')
                                                                                          === 'true',
                                   'MAIN_SHOW_QTY'                                     => gm_get_conf('MAIN_SHOW_QTY')
                                                                                          === 'true',
                                   'MAIN_VIEW_MODE_TILED'                              => gm_get_conf('MAIN_VIEW_MODE_TILED')
                                                                                          === 'true',
                                   'SHOW_MANUFACTURER_IMAGE_LISTING'                   => gm_get_conf('SHOW_MANUFACTURER_IMAGE_LISTING')
                                                                                          === 'true',
                                   'SHOW_PRODUCT_RIBBONS'                              => gm_get_conf('SHOW_PRODUCT_RIBBONS')
                                                                                          === 'true',
                                   'GM_SHOW_FLYOVER'                                   => gm_get_conf('GM_SHOW_FLYOVER')
                                                                                          === 'true',
                                   'SHOW_GALLERY'                                      => gm_get_conf('SHOW_GALLERY')
                                                                                          === 'true',
                                   'SHOW_ZOOM'                                         => gm_get_conf('SHOW_ZOOM')
                                                                                          === 'true',
                                   'CAT_MENU_TOP'                                      => gm_get_conf('CAT_MENU_TOP')
                                                                                          === 'true',
                                   'CAT_MENU_LEFT'                                     => gm_get_conf('CAT_MENU_LEFT')
                                                                                          === 'true',
                                   'SHOW_SUBCATEGORIES'                                => gm_get_conf('SHOW_SUBCATEGORIES')
                                                                                          === 'true',
                                   'CATEGORY_ACCORDION_EFFECT'                         => gm_get_conf('CATEGORY_ACCORDION_EFFECT')
                                                                                          === 'true',
                                   'CATEGORY_UNFOLD_LEVEL'                             => gm_get_conf('CATEGORY_UNFOLD_LEVEL'),
                                   'CATEGORY_UNFOLD_DEFAULT_LEVEL'                     => gm_get_conf('CATEGORY_UNFOLD_DEFAULT_LEVEL'),
                                   'CATEGORY_TOP_SHOW_LEVEL'                           => gm_get_conf('CATEGORY_TOP_SHOW_LEVEL'),
                                   'SHOW_TOP_CURRENCY_SELECTION'                       => gm_get_conf('SHOW_TOP_CURRENCY_SELECTION')
                                                                                          === 'true',
                                   'SHOW_TOP_COUNTRY_SELECTION'                        => gm_get_conf('SHOW_TOP_COUNTRY_SELECTION')
                                                                                          === 'true',
                                   'GM_QUICK_SEARCH'                                   => gm_get_conf('GM_QUICK_SEARCH')
                                                                                          === 'true',
                                   'SHOW_FACEBOOK'                                     => gm_get_conf('SHOW_FACEBOOK')
                                                                                          === 'true',
                                   'SHOW_WHATSAPP'                                     => gm_get_conf('SHOW_WHATSAPP')
                                                                                          === 'true',
                                   'SHOW_TWITTER'                                      => gm_get_conf('SHOW_TWITTER')
                                                                                          === 'true',
                                   'SHOW_PINTEREST'                                    => gm_get_conf('SHOW_PINTEREST')
                                                                                          === 'true',
                                   'GALLERY_LIGHTBOX'                                  => gm_get_conf('GALLERY_LIGHTBOX')
                                                                                          === 'true',
                                   'GM_SHOW_WISHLIST'                                  => gm_get_conf('GM_SHOW_WISHLIST')
                                                                                          === 'true',
                                   'GM_SPECIALS_STARTPAGE'                             => gm_get_conf('GM_SPECIALS_STARTPAGE'),
                                   'GM_NEW_PRODUCTS_STARTPAGE'                         => gm_get_conf('GM_NEW_PRODUCTS_STARTPAGE'),
                                   'GM_RECOMMENDED_PRODUCTS_STARTPAGE'                 => gm_get_conf('GM_RECOMMENDED_PRODUCTS_STARTPAGE'),
                                   'ENABLE_RATING'                                     => gm_get_conf('ENABLE_RATING')
                                                                                          === 'true',
                                   'SHOW_RATING_IN_GRID_AND_LISTING'                   => gm_get_conf('SHOW_RATING_IN_GRID_AND_LISTING')
                                                                                          === 'true',
                                   'SHOW_RATING_AS_TAB'                                => gm_get_conf('SHOW_RATING_AS_TAB')
                                                                                          === 'true',
                                   'ALWAYS_SHOW_CONTINUE_SHOPPING_BUTTON'              => gm_get_conf('ALWAYS_SHOW_CONTINUE_SHOPPING_BUTTON')
                                                                                          === 'true',
                                   'SHOP_ENVIRONMENT'                                  => $this->shopEnvironment,
                                   'STYLE_EDIT_3_LINK'                                 => $this->styleEdit3Link,
                                   'STYLE_EDIT_4_LINK'                                 => $this->styleEdit4Link,
                                   'USER_HAS_ACCESS_TO_STYLE_EDIT'                     => $this->userHasAccessToStyleEditControllers(),
                                   'STYLE_EDIT_SOS_LINK'                               => xtc_href_link('../index.php',
                                                                                                        'style_edit_mode=sos'),
                                   'GM_TELL_A_FRIEND'                                  => gm_get_conf('GM_TELL_A_FRIEND')
                                                                                          === 'true',
                                   'GM_TELL_A_FRIEND_USE_CUSTOMER_EMAIL'               => gm_get_conf('GM_TELL_A_FRIEND_USE_CUSTOMER_EMAIL')
                                                                                          === 'true',
                                   'USE_UPCOMING_PRODUCT_SWIPER_ON_INDEX'              => gm_get_conf('USE_UPCOMING_PRODUCT_SWIPER_ON_INDEX')
                                                                                          === 'true',
                                   'USE_TOP_PRODUCT_SWIPER_ON_INDEX'                   => gm_get_conf('USE_TOP_PRODUCT_SWIPER_ON_INDEX')
                                                                                          === 'true',
                                   'USE_SPECIAL_PRODUCT_SWIPER_ON_INDEX'               => gm_get_conf('USE_SPECIAL_PRODUCT_SWIPER_ON_INDEX')
                                                                                          === 'true',
                                   'USE_NEW_PRODUCT_SWIPER_ON_INDEX'                   => gm_get_conf('USE_NEW_PRODUCT_SWIPER_ON_INDEX')
                                                                                          === 'true',
                                   'USE_SMALLER_IMAGES_FOR_PRODUCTS'                   => gm_get_conf('USE_SMALLER_IMAGES_FOR_PRODUCTS')
                                                                                          === 'true',
                                   'SHOW_PRODUCTS_COUNT'                               => gm_get_conf('SHOW_PRODUCTS_COUNT')
                                                                                          === 'true',
                                   'SHOW_PRODUCTS_MODEL_IN_SHOPPING_CART_AND_WISHLIST' => gm_get_conf('SHOW_PRODUCTS_MODEL_IN_SHOPPING_CART_AND_WISHLIST')
                                                                                          === 'true',
                                   'SHOW_PRODUCTS_MODEL_IN_PRODUCT_DETAILS'            => gm_get_conf('SHOW_PRODUCTS_MODEL_IN_PRODUCT_DETAILS')
                                                                                          === 'true',
                                   'SHOW_PRODUCTS_MODEL_IN_PRODUCT_LISTS'              => gm_get_conf('SHOW_PRODUCTS_MODEL_IN_PRODUCT_LISTS')
                                                                                          === 'true',
                                   'ENABLE_JS_HYPHENATION'                             => gm_get_conf('ENABLE_JS_HYPHENATION')
                                                                                          === 'true',
                                   'ENABLE_LIVE_SEARCH'                                => gm_get_conf('ENABLE_LIVE_SEARCH')
                                                                                          === 'true',
                                   'DISPLAY_OF_PROPERTY_COMBINATION_SELECTION'         => gm_get_conf('DISPLAY_OF_PROPERTY_COMBINATION_SELECTION'),
                                   'STYLE_EDIT_LINK_LAYOUT'                            => $this->determineStyleEditLinkLayout(),
                               ]);
        
        $pageTitle = $this->languageTextManager->get_text('HEADING_TITLE');
        
        return MainFactory::create('AdminPageHttpControllerResponse', $pageTitle, $html);
    }
    
    
    /**
     * This method was copied from admin/includes/functions/general.php function xtc_cfg_pull_down_template_sets.
     * Instead of returning html, this method returns an array containing all information needed.
     */
    protected function templateSelection()
    {
        $templatesArray = $deactivatedTemplatesArray = [];
        $templatesPath  = DIR_FS_CATALOG . 'templates';
        
        $directoryIterator = new IteratorIterator(new DirectoryIterator($templatesPath));
        
        /** @var \DirectoryIterator $directory */
        foreach ($directoryIterator as $directory) {
            // fetches all templates from the shop and prepare $templatesArray with the fetched data
            if ($directory->isDir() && !$directory->isDot()) {
                $templateName = $directory->getFilename();
                
                if ($templateName === CURRENT_TEMPLATE) {
                    $templatesArray[] = ['id' => 'template-' . $templateName, 'text' => $templateName . ' (Template)'];
                    continue;
                }
                $templateSettings = $directory->getPathname() . '/template_settings.php';
                
                if (file_exists($templateSettings)) {
                    include $templateSettings;
                    
                    if (isset($t_template_settings_array)
                        && is_array($t_template_settings_array)
                        && array_key_exists('TEMPLATE_PRESENTATION_VERSION', $t_template_settings_array)
                        && $t_template_settings_array['TEMPLATE_PRESENTATION_VERSION'] >= 2.0) {
                        $templatesArray[] = [
                            'id'   => 'template-' . $templateName,
                            'text' => $templateName . ' (Template)'
                        ];
                    }
                }
            }
        }
        
        $themesPath = DIR_FS_CATALOG . 'themes';
        if (is_dir($themesPath)) {
            /** @var \ThemeService $themeService */
            $themeService = StaticGXCoreLoader::getService('Theme');
            
            $availableThemes = $themeService->getAvailableThemes(ThemeDirectoryRoot::create(new ExistingDirectory($themesPath)));
            
            DependencyInjector::inject();
            
            /** @var \ThemeName $availableTheme */
            foreach ($availableThemes as $availableTheme) {
                $themeJson = get_theme_json_from_theme_path($themesPath . DIRECTORY_SEPARATOR
                                                            . $availableTheme->getName());
                
                if ($themeJson !== null && !isset($themeJson->preview)) {
                    
                    $templatesArray[] = [
                        'id'   => 'theme-' . $availableTheme->getName(),
                        'text' => $availableTheme->getName() . ' (Theme)'
                    ];
                }
            }
        }
        sort($templatesArray);
        
        $themeControl = StaticGXCoreLoader::getThemeControl();
        if ($themeControl->isThemeSystemActive()) {
            $default = 'theme-' . CURRENT_THEME;
        } else {
            $default = 'template-' . CURRENT_TEMPLATE;
        }
        
        return [
            'templates' => $templatesArray,
            'default'   => $default
        ];
    }
    
    
    /**
     * @return bool
     */
    protected function userHasAccessToStyleEditControllers(): bool
    {
        /** @var AdminAccessService $adminAccessService */
        $adminAccessService = StaticGXCoreLoader::getService('AdminAccess');
        
        try {
            // StyleEdit 3 & 4 are in the same access group
            $controllerName = new NonEmptyStringType('StyleEdit4Authentication');
            $customerIdInt  = (int)$_SESSION['customer_id'];
            $customerId     = new IdType($customerIdInt);
            
            return $adminAccessService->checkWritingPermissionForController($controllerName, $customerId)
                   && $adminAccessService->checkReadingPermissionForController($controllerName, $customerId);
        } catch (Exception $exception) {
            return true;
        }
    }
    
    
    /**
     * @return string
     */
    protected function determineStyleEditLinkLayout(): string
    {
        if (StaticGXCoreLoader::getThemeControl()->isThemeSystemActive()) {
            return 'ACTIVE_THEME';
        }
        
        return file_exists(SHOP_ROOT
                           . 'StyleEdit3/bootstrap.inc.php') ? 'ACTIVE_TEMPLATE_WITH_STYLEEDIT3' : 'ACTIVE_TEMPLATE_WITHOUT_STYLEEDIT3';
    }
    
    
    /**
     * Save shop key
     *
     * @return RedirectHttpControllerResponse
     */
    public function actionStore()
    {
        $this->_store('MAIN_SHOW_QTY_INFO', $this->_getPostData('MAIN_SHOW_QTY_INFO'));
        $this->_store('MAIN_SHOW_ATTRIBUTES', $this->_getPostData('MAIN_SHOW_ATTRIBUTES'));
        $this->_store('MAIN_SHOW_GRADUATED_PRICES', $this->_getPostData('MAIN_SHOW_GRADUATED_PRICES'));
        $this->_store('SHOW_MANUFACTURER_IMAGE_PRODUCT_DETAILS',
                      $this->_getPostData('SHOW_MANUFACTURER_IMAGE_PRODUCT_DETAILS'));
        $this->_store('SHOW_ADDITIONAL_FIELDS_PRODUCT_DETAILS',
                      $this->_getPostData('SHOW_ADDITIONAL_FIELDS_PRODUCT_DETAILS'));
        $this->_store('MAIN_SHOW_QTY', $this->_getPostData('MAIN_SHOW_QTY'));
        $this->_store('MAIN_VIEW_MODE_TILED', $this->_getPostData('MAIN_VIEW_MODE_TILED'));
        $this->_store('SHOW_MANUFACTURER_IMAGE_LISTING', $this->_getPostData('SHOW_MANUFACTURER_IMAGE_LISTING'));
        $this->_store('SHOW_PRODUCT_RIBBONS', $this->_getPostData('SHOW_PRODUCT_RIBBONS'));
        $this->_store('GM_SHOW_FLYOVER', $this->_getPostData('GM_SHOW_FLYOVER'));
        $this->_store('SHOW_GALLERY', $this->_getPostData('SHOW_GALLERY'));
        $this->_store('SHOW_ZOOM', $this->_getPostData('SHOW_ZOOM'));
        $this->_store('CAT_MENU_TOP', $this->_getPostData('CAT_MENU_TOP'));
        $this->_store('CAT_MENU_LEFT', $this->_getPostData('CAT_MENU_LEFT'));
        $this->_store('SHOW_SUBCATEGORIES', $this->_getPostData('SHOW_SUBCATEGORIES'));
        $this->_store('CATEGORY_ACCORDION_EFFECT', $this->_getPostData('CATEGORY_ACCORDION_EFFECT'));
        $this->_store('CATEGORY_UNFOLD_DEFAULT_LEVEL', (int)$this->_getPostData('CATEGORY_UNFOLD_DEFAULT_LEVEL'));
        $this->_store('CATEGORY_UNFOLD_LEVEL', (int)$this->_getPostData('CATEGORY_UNFOLD_LEVEL'));
        $this->_store('CATEGORY_TOP_SHOW_LEVEL', (int)$this->_getPostData('CATEGORY_TOP_SHOW_LEVEL'));
        $this->_store('SHOW_TOP_CURRENCY_SELECTION', $this->_getPostData('SHOW_TOP_CURRENCY_SELECTION'));
        $this->_store('SHOW_TOP_COUNTRY_SELECTION', $this->_getPostData('SHOW_TOP_COUNTRY_SELECTION'));
        $this->_store('GM_QUICK_SEARCH', $this->_getPostData('GM_QUICK_SEARCH'));
        $this->_store('SHOW_FACEBOOK', $this->_getPostData('SHOW_FACEBOOK'));
        $this->_store('SHOW_WHATSAPP', $this->_getPostData('SHOW_WHATSAPP'));
        $this->_store('SHOW_TWITTER', $this->_getPostData('SHOW_TWITTER'));
        $this->_store('SHOW_PINTEREST', $this->_getPostData('SHOW_PINTEREST'));
        $this->_store('GALLERY_LIGHTBOX', $this->_getPostData('GALLERY_LIGHTBOX'));
        $this->_store('GM_SHOW_WISHLIST', $this->_getPostData('GM_SHOW_WISHLIST'));
        $this->_store('GM_SPECIALS_STARTPAGE', (int)$this->_getPostData('GM_SPECIALS_STARTPAGE'));
        $this->_store('GM_NEW_PRODUCTS_STARTPAGE', (int)$this->_getPostData('GM_NEW_PRODUCTS_STARTPAGE'));
        $this->_store('GM_RECOMMENDED_PRODUCTS_STARTPAGE',
                      (int)$this->_getPostData('GM_RECOMMENDED_PRODUCTS_STARTPAGE'));
        $this->_store('ENABLE_RATING', $this->_getPostData('ENABLE_RATING'));
        $this->_store('SHOW_RATING_IN_GRID_AND_LISTING', $this->_getPostData('SHOW_RATING_IN_GRID_AND_LISTING'));
        $this->_store('SHOW_RATING_AS_TAB', $this->_getPostData('SHOW_RATING_AS_TAB'));
        $this->_store('ALWAYS_SHOW_CONTINUE_SHOPPING_BUTTON',
                      $this->_getPostData('ALWAYS_SHOW_CONTINUE_SHOPPING_BUTTON'));
        $this->_store('GM_TELL_A_FRIEND', $this->_getPostData('GM_TELL_A_FRIEND'));
        $this->_store('USE_UPCOMING_PRODUCT_SWIPER_ON_INDEX',
                      $this->_getPostData('USE_UPCOMING_PRODUCT_SWIPER_ON_INDEX'));
        $this->_store('USE_TOP_PRODUCT_SWIPER_ON_INDEX', $this->_getPostData('USE_TOP_PRODUCT_SWIPER_ON_INDEX'));
        $this->_store('USE_SPECIAL_PRODUCT_SWIPER_ON_INDEX',
                      $this->_getPostData('USE_SPECIAL_PRODUCT_SWIPER_ON_INDEX'));
        $this->_store('USE_NEW_PRODUCT_SWIPER_ON_INDEX', $this->_getPostData('USE_NEW_PRODUCT_SWIPER_ON_INDEX'));
        $this->_store('USE_SMALLER_IMAGES_FOR_PRODUCTS', $this->_getPostData('USE_SMALLER_IMAGES_FOR_PRODUCTS'));
        $this->_store('SHOW_PRODUCTS_COUNT', $this->_getPostData('SHOW_PRODUCTS_COUNT'));
        $this->_store('SHOW_PRODUCTS_MODEL_IN_SHOPPING_CART_AND_WISHLIST',
                      $this->_getPostData('SHOW_PRODUCTS_MODEL_IN_SHOPPING_CART_AND_WISHLIST'));
        $this->_store('SHOW_PRODUCTS_MODEL_IN_PRODUCT_DETAILS',
                      $this->_getPostData('SHOW_PRODUCTS_MODEL_IN_PRODUCT_DETAILS'));
        $this->_store('SHOW_PRODUCTS_MODEL_IN_PRODUCT_LISTS',
                      $this->_getPostData('SHOW_PRODUCTS_MODEL_IN_PRODUCT_LISTS'));
        $this->_store('ENABLE_JS_HYPHENATION', $this->_getPostData('ENABLE_JS_HYPHENATION'));
        $this->_store('ENABLE_LIVE_SEARCH', $this->_getPostData('ENABLE_LIVE_SEARCH'));
        $this->_store('GM_TELL_A_FRIEND_USE_CUSTOMER_EMAIL', $this->_getPostData('GM_TELL_A_FRIEND_USE_CUSTOMER_EMAIL'));
        $this->_store('DISPLAY_OF_PROPERTY_COMBINATION_SELECTION', $this->_getPostData('DISPLAY_OF_PROPERTY_COMBINATION_SELECTION'));
        
        $this->_updateTheme();
        
        $url = xtc_href_link('admin.php', 'do=TemplateConfiguration');
        
        return MainFactory::create('RedirectHttpControllerResponse', $url);
    }
    
    
    protected function _updateTheme()
    {
        $currentTemplate       = $_POST['CURRENT_THEME'];
        $isTheme               = strpos($currentTemplate, 'theme-') === 0;
        $prefix                = $isTheme ? 'theme-' : 'template-';
        $sanitizedTemplateName = str_replace($prefix, '', $currentTemplate);
        
        if ($isTheme) {
            /** @var \ThemeService $themeService */
            $themeService = StaticGXCoreLoader::getService('Theme');
            
            $themeId = ThemeId::create($sanitizedTemplateName);
            
            $source      = ThemeDirectoryRoot::create(new ExistingDirectory(DIR_FS_CATALOG . 'themes'));
            $destination = ThemeDirectoryRoot::create(new ExistingDirectory(DIR_FS_CATALOG
                                                                            . StaticGXCoreLoader::getThemeControl()
                                                                                ->getPublishedThemePath()));
            $settings    = ThemeSettings::create($source, $destination);
            
            $themeService->buildTemporaryTheme($themeId, $settings);
            
            try {
                $themeService->activateTheme($sanitizedTemplateName);
            } catch (Exception $e) {
                $coo_logger = LogControl::get_instance();
                $coo_logger->error($e->getMessage() . '- ' . $e->getFile() . ' - ' . $e->getLine());
                echo $e->getMessage();
            }
        } else {
            $templateQuery = 'UPDATE `gx_configurations` SET `value` = "' . $sanitizedTemplateName
                             . '" WHERE `key` = "configuration/CURRENT_TEMPLATE"';
            $themeQuery    = 'UPDATE `gx_configurations` SET `value` = "" WHERE `key` = "configuration/CURRENT_THEME"';
            xtc_db_query($templateQuery);
            xtc_db_query($themeQuery);
        }
    
        $coo_cache_control     = MainFactory::create_object('CacheControl');
        $coo_cache_control->clear_templates_c();
        $coo_cache_control->clear_template_cache();
        $coo_cache_control->clear_css_cache();
    }
    
    
    /**
     * Update the template configuration values in the database
     *
     * @param string $p_key
     * @param string $p_value
     */
    protected function _store($p_key, $p_value)
    {
        if ($p_value !== null) {
            $key   = "gm_configuration/$p_key";
            $this->configurationService->save(Configuration::create($key, $p_value));
        }
    }
    
    
    protected function _checkEnvironment()
    {
        include_once DIR_FS_CATALOG . 'GXModules/Gambio/StyleEdit/Api/Storage/StyleEditWelcomeStorage.php';
        $welcome_storage           = new \Gambio\StyleEdit\Api\Storage\StyleEditWelcomeStorage;
        $userHasSeenWelcomeMessage = $welcome_storage->welcomeStatusSeenForCustomer(new IdType($_SESSION['customer_id']));
        
        if (StyleEditServiceFactory::service() instanceof StyleEdit4Service) {
            
            $this->shopEnvironment = 'StyleEdit4';
        } elseif (is_dir(DIR_FS_CATALOG . 'StyleEdit3/templates/' . StaticGXCoreLoader::getThemeControl()
                             ->getCurrentTheme())
                  || is_dir(DIR_FS_CATALOG . StaticGXCoreLoader::getThemeControl()->getPublishedThemePath()
                            . '/styles/styleedit')) {
            $this->shopEnvironment = 'StyleEdit3';
            $this->styleEditLink   = xtc_href_link('admin.php', 'do=StyleEdit3Authentication');
        } elseif (is_dir(DIR_FS_CATALOG . 'StyleEdit/') !== false && gm_get_env_info('TEMPLATE_VERSION') < 3) {
            $this->shopEnvironment = 'StyleEdit';
            $this->styleEditLink   = xtc_href_link('../index.php', 'style_edit_mode=edit');
            
            // add StyleEdit3 link, if the honeygrid template exists
            if (is_dir(DIR_FS_CATALOG . 'StyleEdit3/templates/Honeygrid')) {
                $this->styleEdit3Link = xtc_href_link('admin.php', 'do=StyleEdit3Authentication');
            }
        } else {
            if (!isset($_GET['force_config']) || $_GET['force_config'] !== 'true') {
                $this->shopEnvironment = 'noStyleEdit';
            } elseif (gm_get_env_info('TEMPLATE_VERSION') >= 3) {
                $this->shopEnvironment = 'forceStyleEdit';
                $this->styleEditLink   = '#';
            }
        }
        
        $this->styleEdit3Link = xtc_href_link('admin.php', 'do=StyleEdit3Authentication');
        $this->styleEdit4Link = $userHasSeenWelcomeMessage ? xtc_href_link('admin.php',
                                                                           'do=StyleEdit4Authentication') : xtc_href_link('admin.php',
                                                                                                                          'do=StyleEdit4Authentication&welcome=1');
    }
}
